<?php

namespace App\Http\Controllers;

use App\Models\Lease;
use App\Models\SecurityDepositPayment;
use App\Models\Shop;
use App\Models\Tenant;
use Illuminate\Http\Request;
use Inertia\Inertia;

class LeaseController extends Controller
{
    public function index(Request $request)
    {
        $leases = Lease::with(['shop.owner', 'tenant', 'invoices' => function ($q) {
                $q->select('id', 'lease_id', 'invoice_number', 'month', 'total_amount', 'status')
                  ->withSum('payments', 'amount_paid');
            }])
            ->when($request->status, fn($q, $s) => $q->where('status', $s))
            ->when($request->search, fn($q, $s) => $q->whereHas('tenant', fn($tq) => $tq->where('name', 'like', "%{$s}%"))->orWhereHas('shop', fn($sq) => $sq->where('shop_number', 'like', "%{$s}%")))
            ->latest()
            ->paginate(15)
            ->withQueryString();

        return Inertia::render('Leases/Index', [
            'leases' => $leases,
            'filters' => $request->only('search', 'status'),
        ]);
    }

    public function create()
    {
        $shops = Shop::where('status', 'vacant')->with('owner')->orderBy('shop_number')->get();
        $tenants = Tenant::orderBy('name')->get(['id', 'name', 'phone', 'cnic']);

        return Inertia::render('Leases/Create', [
            'shops' => $shops,
            'tenants' => $tenants,
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'shop_id' => 'required|exists:shops,id',
            'tenant_id' => 'required|exists:tenants,id',
            'start_date' => 'required|date',
            'grace_days' => 'nullable|integer|min:0',
            'end_date' => 'nullable|date|after:start_date',
            'agreed_rent' => 'required|numeric|min:0',
            'annual_increase_type' => 'required|in:percentage,fixed',
            'annual_increase_value' => 'nullable|numeric|min:0',
            'initial_rent' => 'nullable|numeric|min:0',
            'initial_rent_months' => 'nullable|integer|min:1',
            'agreed_security_deposit' => 'nullable|numeric|min:0',
            'security_deposit_paid' => 'nullable|numeric|min:0',
            'maintenance_fee' => 'nullable|numeric|min:0',
            'electricity_enabled' => 'boolean',
            'electricity_rate_per_unit' => 'nullable|numeric|min:0',
            'agreement_file' => 'nullable|file|mimes:pdf|max:10240',
            'agreement_notes' => 'nullable|string',
            'commission_type' => 'required|in:none,full_rent,half_rent,fixed',
            'commission_amount' => 'nullable|numeric|min:0',
            'commission_to' => 'nullable|string|max:255',
            'commission_phone' => 'nullable|string|max:20',
            'commission_remarks' => 'nullable|string',
        ]);

        if ($request->hasFile('agreement_file')) {
            $validated['agreement_file'] = $request->file('agreement_file')->store('agreements', 'public');
        }

        $validated['grace_days'] = $validated['grace_days'] ?? 0;
        $validated['agreed_security_deposit'] = $validated['agreed_security_deposit'] ?? 0;
        $validated['security_deposit_paid'] = $validated['security_deposit_paid'] ?? 0;

        $lease = Lease::create($validated);

        // Update shop status
        Shop::where('id', $validated['shop_id'])->update(['status' => 'occupied']);

        // Record initial security deposit payment if any
        if ($validated['security_deposit_paid'] > 0) {
            SecurityDepositPayment::create([
                'lease_id' => $lease->id,
                'amount' => $validated['security_deposit_paid'],
                'payment_date' => now(),
                'payment_method' => 'cash',
                'remarks' => 'Initial security deposit',
            ]);
        }

        return redirect()->route('leases.index')->with('success', 'Lease created successfully.');
    }

    public function show(Lease $lease)
    {
        $lease->load(['shop.owner', 'tenant', 'invoices.payments', 'securityDepositPayments', 'meterReadings']);
        $lease->append('current_rent');

        return Inertia::render('Leases/Show', ['lease' => $lease]);
    }

    public function edit(Lease $lease)
    {
        $shops = Shop::where('status', 'vacant')->orWhere('id', $lease->shop_id)->with('owner')->orderBy('shop_number')->get();
        $tenants = Tenant::orderBy('name')->get(['id', 'name', 'phone', 'cnic']);

        return Inertia::render('Leases/Edit', [
            'lease' => $lease,
            'shops' => $shops,
            'tenants' => $tenants,
        ]);
    }

    public function update(Request $request, Lease $lease)
    {
        $validated = $request->validate([
            'start_date' => 'required|date',
            'grace_days' => 'nullable|integer|min:0',
            'end_date' => 'nullable|date|after:start_date',
            'agreed_rent' => 'required|numeric|min:0',
            'annual_increase_type' => 'required|in:percentage,fixed',
            'annual_increase_value' => 'nullable|numeric|min:0',
            'initial_rent' => 'nullable|numeric|min:0',
            'initial_rent_months' => 'nullable|integer|min:1',
            'agreed_security_deposit' => 'nullable|numeric|min:0',
            'maintenance_fee' => 'nullable|numeric|min:0',
            'electricity_enabled' => 'boolean',
            'electricity_rate_per_unit' => 'nullable|numeric|min:0',
            'agreement_notes' => 'nullable|string',
            'commission_type' => 'required|in:none,full_rent,half_rent,fixed',
            'commission_amount' => 'nullable|numeric|min:0',
            'commission_to' => 'nullable|string|max:255',
            'commission_phone' => 'nullable|string|max:20',
            'commission_remarks' => 'nullable|string',
        ]);

        if ($request->hasFile('agreement_file')) {
            $validated['agreement_file'] = $request->file('agreement_file')->store('agreements', 'public');
        }

        $lease->update($validated);

        return redirect()->route('leases.show', $lease)->with('success', 'Lease updated successfully.');
    }

    public function destroy(Lease $lease)
    {
        Shop::where('id', $lease->shop_id)->update(['status' => 'vacant']);
        $lease->delete();
        return redirect()->route('leases.index')->with('success', 'Lease deleted successfully.');
    }

    public function terminate(Lease $lease)
    {
        $lease->update(['status' => 'terminated', 'end_date' => now()]);
        Shop::where('id', $lease->shop_id)->update(['status' => 'vacant']);

        return redirect()->route('leases.show', $lease)->with('success', 'Lease terminated successfully.');
    }

    public function recordDeposit(Request $request, Lease $lease)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:1',
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:cash,bank_transfer',
            'proof_file' => 'nullable|file|max:5120',
            'remarks' => 'nullable|string',
            'received_by' => 'nullable|required_if:payment_method,cash|string|max:255',
        ]);

        if ($request->hasFile('proof_file')) {
            $validated['proof_file'] = $request->file('proof_file')->store('deposit-proofs', 'public');
        }

        $validated['lease_id'] = $lease->id;
        SecurityDepositPayment::create($validated);

        $lease->increment('security_deposit_paid', $validated['amount']);

        return redirect()->route('leases.show', $lease)->with('success', 'Security deposit recorded.');
    }

    public function payCommission(Lease $lease)
    {
        if ($lease->commission_paid) {
            return redirect()->route('leases.show', $lease)->with('error', 'Commission already paid.');
        }

        if ($lease->commission_type === 'none' || $lease->commission_amount <= 0) {
            return redirect()->route('leases.show', $lease)->with('error', 'No commission to pay.');
        }

        $lease->update([
            'commission_paid' => true,
            'commission_paid_date' => now(),
        ]);

        return redirect()->route('leases.show', $lease)->with('success', 'Commission marked as paid.');
    }

    public function commissionReceipt(Lease $lease)
    {
        $lease->load(['shop.owner', 'tenant']);

        $settings = \App\Models\Setting::all()->pluck('value', 'key');

        return Inertia::render('Leases/CommissionReceipt', [
            'lease' => $lease,
            'settings' => $settings,
        ]);
    }

    public function renew(Request $request, Lease $lease)
    {
        $validated = $request->validate([
            'start_date' => 'required|date',
            'grace_days' => 'nullable|integer|min:0',
            'end_date' => 'nullable|date|after:start_date',
            'agreed_rent' => 'required|numeric|min:0',
            'annual_increase_type' => 'required|in:percentage,fixed',
            'annual_increase_value' => 'nullable|numeric|min:0',
        ]);

        // Terminate the old lease
        $lease->update(['status' => 'expired', 'end_date' => $lease->end_date ?? now()]);

        // Create new lease with old lease data + new terms
        $newLease = Lease::create([
            'shop_id' => $lease->shop_id,
            'tenant_id' => $lease->tenant_id,
            'start_date' => $validated['start_date'],
            'grace_days' => $validated['grace_days'] ?? 0,
            'end_date' => $validated['end_date'],
            'agreed_rent' => $validated['agreed_rent'],
            'annual_increase_type' => $validated['annual_increase_type'],
            'annual_increase_value' => $validated['annual_increase_value'] ?? 0,
            'maintenance_fee' => $lease->maintenance_fee,
            'electricity_enabled' => $lease->electricity_enabled,
            'electricity_rate_per_unit' => $lease->electricity_rate_per_unit,
            'agreed_security_deposit' => $lease->agreed_security_deposit,
            'security_deposit_paid' => $lease->security_deposit_paid,
            'status' => 'active',
        ]);

        return redirect()->route('leases.show', $newLease)->with('success', 'Lease renewed successfully! New lease created.');
    }
}
