<?php

namespace App\Http\Controllers;

use App\Models\Lease;
use App\Models\MeterReading;
use App\Models\Setting;
use Illuminate\Http\Request;
use Inertia\Inertia;

class MeterReadingController extends Controller
{
    public function index(Request $request)
    {
        $readings = MeterReading::with(['lease.shop', 'lease.tenant'])
            ->when($request->month, fn($q, $m) => $q->where('month', $m))
            ->latest()
            ->paginate(20)
            ->withQueryString();

        $leases = Lease::active()->where('electricity_enabled', true)->with(['shop', 'tenant'])->get();
        $defaultRate = (float) Setting::get('default_electricity_rate', 25);

        return Inertia::render('MeterReadings/Index', [
            'readings' => $readings,
            'filters' => $request->only('month'),
            'leases' => $leases,
            'defaultRate' => $defaultRate,
        ]);
    }

    public function create()
    {
        $leases = Lease::active()->where('electricity_enabled', true)->with(['shop', 'tenant'])->get();
        $defaultRate = Setting::get('default_electricity_rate', 25);

        return Inertia::render('MeterReadings/Create', [
            'leases' => $leases,
            'defaultRate' => (float) $defaultRate,
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'lease_id' => 'required|exists:leases,id',
            'month' => 'required|string|size:7',
            'previous_reading' => 'required|numeric|min:0',
            'current_reading' => 'required|numeric|gte:previous_reading',
            'rate_per_unit' => 'required|numeric|min:0',
        ]);

        $validated['units_consumed'] = $validated['current_reading'] - $validated['previous_reading'];
        $validated['total_amount'] = $validated['units_consumed'] * $validated['rate_per_unit'];

        MeterReading::create($validated);

        return redirect()->route('meter-readings.index')->with('success', 'Meter reading recorded.');
    }

    public function edit(MeterReading $meterReading)
    {
        $meterReading->load(['lease.shop', 'lease.tenant']);

        return Inertia::render('MeterReadings/Edit', [
            'reading' => $meterReading,
        ]);
    }

    public function update(Request $request, MeterReading $meterReading)
    {
        $validated = $request->validate([
            'previous_reading' => 'required|numeric|min:0',
            'current_reading' => 'required|numeric|gte:previous_reading',
            'rate_per_unit' => 'required|numeric|min:0',
        ]);

        $validated['units_consumed'] = $validated['current_reading'] - $validated['previous_reading'];
        $validated['total_amount'] = $validated['units_consumed'] * $validated['rate_per_unit'];

        $meterReading->update($validated);

        return redirect()->route('meter-readings.index')->with('success', 'Meter reading updated.');
    }

    public function destroy(MeterReading $meterReading)
    {
        $meterReading->delete();
        return redirect()->route('meter-readings.index')->with('success', 'Meter reading deleted.');
    }
}
