<?php

namespace App\Http\Controllers;

use App\Models\Owner;
use App\Models\OwnerPayment;
use App\Models\OwnerPaymentItem;
use App\Models\Payment;
use App\Models\Setting;
use App\Models\Shop;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class OwnerPaymentController extends Controller
{
    public function index(Request $request)
    {
        // Build owner summary: total collected from tenants vs total paid to owner
        $ownerIds = $request->owner_ids ? explode(',', $request->owner_ids) : [];

        $ownersQuery = Owner::with('shops:id,owner_id,shop_number')
            ->has('shops')
            ->orderBy('name');

        $owners = $ownersQuery->get();

        // Calculate per-owner summaries
        $ownerSummaries = [];
        foreach ($owners as $owner) {
            $shopIds = $owner->shops->pluck('id')->toArray();

            // Total collected from tenants (payments on invoices of leases on these shops)
            $totalCollected = Payment::whereHas('invoice.lease', function ($q) use ($shopIds) {
                $q->whereIn('shop_id', $shopIds);
            })->sum('amount_paid');

            // Total paid to this owner
            $totalPaidToOwner = OwnerPayment::where('owner_id', $owner->id)->sum('total_amount');

            $pending = $totalCollected - $totalPaidToOwner;

            $ownerSummaries[] = [
                'id' => $owner->id,
                'name' => $owner->name,
                'phone' => $owner->phone,
                'shops_count' => $owner->shops->count(),
                'shop_numbers' => $owner->shops->pluck('shop_number')->toArray(),
                'total_collected' => round($totalCollected, 2),
                'total_paid' => round($totalPaidToOwner, 2),
                'pending' => round($pending, 2),
            ];
        }

        // Filter payment records
        $payments = OwnerPayment::with(['owner', 'items.shop'])
            ->when(count($ownerIds) > 0, fn($q) => $q->whereIn('owner_id', $ownerIds))
            ->when($request->status === 'has_pending', function ($q) use ($ownerSummaries) {
                $pendingOwnerIds = collect($ownerSummaries)->where('pending', '>', 0)->pluck('id')->toArray();
                $q->whereIn('owner_id', $pendingOwnerIds);
            })
            ->latest()
            ->paginate(15)
            ->withQueryString();

        // Overall totals
        $grandCollected = array_sum(array_column($ownerSummaries, 'total_collected'));
        $grandPaid = array_sum(array_column($ownerSummaries, 'total_paid'));
        $grandPending = array_sum(array_column($ownerSummaries, 'pending'));

        return Inertia::render('OwnerPayments/Index', [
            'payments' => $payments,
            'owners' => $owners->map(fn($o) => ['id' => $o->id, 'name' => $o->name]),
            'ownerSummaries' => $ownerSummaries,
            'grandTotals' => [
                'collected' => round($grandCollected, 2),
                'paid' => round($grandPaid, 2),
                'pending' => round($grandPending, 2),
            ],
            'filters' => $request->only('owner_ids', 'status'),
        ]);
    }

    public function create(Request $request)
    {
        $owners = Owner::with(['shops' => function ($q) {
            $q->select('id', 'owner_id', 'shop_number');
        }, 'shops.activeLease:id,shop_id,agreed_rent'])->orderBy('name')->get();

        // Calculate pending per owner
        $ownerData = [];
        foreach ($owners as $owner) {
            $shopIds = $owner->shops->pluck('id')->toArray();

            $totalCollected = Payment::whereHas('invoice.lease', function ($q) use ($shopIds) {
                $q->whereIn('shop_id', $shopIds);
            })->sum('amount_paid');

            $totalPaid = OwnerPayment::where('owner_id', $owner->id)->sum('total_amount');

            // Per-shop collected
            $shopData = [];
            foreach ($owner->shops as $shop) {
                $shopCollected = Payment::whereHas('invoice.lease', function ($q) use ($shop) {
                    $q->where('shop_id', $shop->id);
                })->sum('amount_paid');

                $shopPaid = OwnerPaymentItem::whereHas('ownerPayment', function ($q) use ($owner) {
                    $q->where('owner_id', $owner->id);
                })->where('shop_id', $shop->id)->sum('amount');

                $shopData[] = [
                    'id' => $shop->id,
                    'shop_number' => $shop->shop_number,
                    'agreed_rent' => $shop->activeLease ? $shop->activeLease->agreed_rent : 0,
                    'has_lease' => !!$shop->activeLease,
                    'collected' => round($shopCollected, 2),
                    'paid' => round($shopPaid, 2),
                    'pending' => round($shopCollected - $shopPaid, 2),
                ];
            }

            $ownerData[] = [
                'id' => $owner->id,
                'name' => $owner->name,
                'phone' => $owner->phone,
                'bank_name' => $owner->bank_name,
                'bank_account_no' => $owner->bank_account_no,
                'shops' => $shopData,
                'total_collected' => round($totalCollected, 2),
                'total_paid' => round($totalPaid, 2),
                'pending' => round($totalCollected - $totalPaid, 2),
            ];
        }

        return Inertia::render('OwnerPayments/Create', [
            'owners' => $ownerData,
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'owner_id' => 'required|exists:owners,id',
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:cash,bank_transfer',
            'proof_file' => 'nullable|file|max:5120',
            'owner_signature_taken' => 'boolean',
            'remarks' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.shop_id' => 'required|exists:shops,id',
            'items.*.month' => 'required|string|size:7',
            'items.*.amount' => 'required|numeric|min:0',
            'items.*.remarks' => 'nullable|string',
        ]);

        if ($request->hasFile('proof_file')) {
            $validated['proof_file'] = $request->file('proof_file')->store('owner-payment-proofs', 'public');
        }

        $totalAmount = collect($validated['items'])->sum('amount');

        $payment = OwnerPayment::create([
            'owner_id' => $validated['owner_id'],
            'voucher_number' => OwnerPayment::generateVoucherNumber(),
            'payment_date' => $validated['payment_date'],
            'total_amount' => $totalAmount,
            'payment_method' => $validated['payment_method'],
            'proof_file' => $validated['proof_file'] ?? null,
            'owner_signature_taken' => $validated['owner_signature_taken'] ?? false,
            'remarks' => $validated['remarks'] ?? null,
        ]);

        foreach ($validated['items'] as $item) {
            OwnerPaymentItem::create([
                'owner_payment_id' => $payment->id,
                'shop_id' => $item['shop_id'],
                'month' => $item['month'],
                'amount' => $item['amount'],
                'remarks' => $item['remarks'] ?? null,
            ]);
        }

        return redirect()->route('owner-payments.index')->with('success', 'Owner payment recorded.');
    }

    public function show(OwnerPayment $ownerPayment)
    {
        $ownerPayment->load(['owner', 'items.shop']);

        return Inertia::render('OwnerPayments/Show', ['payment' => $ownerPayment]);
    }

    public function print(OwnerPayment $ownerPayment)
    {
        $ownerPayment->load(['owner', 'items.shop']);
        $settings = Setting::all()->pluck('value', 'key');

        return Inertia::render('OwnerPayments/Print', [
            'payment' => $ownerPayment,
            'settings' => $settings,
        ]);
    }

    public function destroy(OwnerPayment $ownerPayment)
    {
        $ownerPayment->delete();
        return redirect()->route('owner-payments.index')->with('success', 'Owner payment deleted.');
    }
}
