<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\Lease;
use App\Models\Owner;
use App\Models\Payment;
use App\Models\Shop;
use Illuminate\Http\Request;
use Inertia\Inertia;

class ReportController extends Controller
{
    public function index()
    {
        return Inertia::render('Reports/Index');
    }

    public function paymentTracker(Request $request)
    {
        $month = $request->get('month', now()->format('Y-m'));

        $shops = Shop::with(['owner', 'activeLease.tenant', 'activeLease.invoices' => fn($q) => $q->where('month', $month)])
            ->whereHas('activeLease')
            ->get()
            ->map(function ($shop) use ($month) {
                $invoice = $shop->activeLease?->invoices->first();
                $totalPaid = $invoice ? $invoice->payments()->sum('amount_paid') : 0;

                return [
                    'shop' => $shop,
                    'invoice' => $invoice,
                    'total_paid' => (float) $totalPaid,
                    'status' => $invoice
                        ? ($invoice->status === 'paid' ? 'paid' : ($invoice->status === 'overdue' ? 'overdue' : 'pending'))
                        : 'no_invoice',
                ];
            });

        return Inertia::render('Reports/PaymentTracker', [
            'shops' => $shops,
            'month' => $month,
        ]);
    }

    public function collection(Request $request)
    {
        $month = $request->get('month', now()->format('Y-m'));

        $invoices = Invoice::with(['lease.shop.owner', 'lease.tenant', 'payments'])
            ->where('month', $month)
            ->get();

        $totalExpected = $invoices->sum('total_amount');
        $totalCollected = $invoices->sum(fn($inv) => $inv->payments->sum('amount_paid'));

        return Inertia::render('Reports/Collection', [
            'invoices' => $invoices,
            'month' => $month,
            'totalExpected' => (float) $totalExpected,
            'totalCollected' => (float) $totalCollected,
            'pending' => (float) ($totalExpected - $totalCollected),
        ]);
    }

    public function ownerWise(Request $request)
    {
        $owners = Owner::with(['shops.activeLease.invoices.payments', 'ownerPayments'])
            ->get()
            ->map(function ($owner) {
                $totalRentCollected = 0;
                $totalPaidToOwner = $owner->ownerPayments->sum('total_amount');

                foreach ($owner->shops as $shop) {
                    if ($shop->activeLease) {
                        foreach ($shop->activeLease->invoices as $invoice) {
                            $totalRentCollected += $invoice->payments->sum('amount_paid');
                        }
                    }
                }

                return [
                    'owner' => $owner,
                    'total_shops' => $owner->shops->count(),
                    'total_rent_collected' => (float) $totalRentCollected,
                    'total_paid_to_owner' => (float) $totalPaidToOwner,
                    'pending_to_owner' => (float) ($totalRentCollected - $totalPaidToOwner),
                ];
            });

        return Inertia::render('Reports/OwnerWise', [
            'owners' => $owners,
        ]);
    }

    public function tenantDues(Request $request)
    {
        $leases = Lease::active()
            ->with(['shop', 'tenant', 'invoices.payments'])
            ->get()
            ->map(function ($lease) {
                $totalDues = 0;
                foreach ($lease->invoices as $invoice) {
                    $paid = $invoice->payments->sum('amount_paid');
                    $totalDues += ($invoice->total_amount - $paid);
                }

                $securityDue = (float) $lease->agreed_security_deposit - (float) $lease->security_deposit_paid;

                return [
                    'lease' => $lease,
                    'rent_dues' => (float) max(0, $totalDues),
                    'security_deposit_due' => (float) max(0, $securityDue),
                    'total_dues' => (float) max(0, $totalDues) + (float) max(0, $securityDue),
                ];
            })
            ->filter(fn($item) => $item['total_dues'] > 0)
            ->values();

        return Inertia::render('Reports/TenantDues', [
            'leases' => $leases,
        ]);
    }
}
