import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { Head, Link } from '@inertiajs/react';
import { formatCurrency, formatDate } from '@/utils';

const StatCard = ({ icon, label, value, color = 'blue' }) => {
    const colors = {
        blue: 'bg-blue-50 text-blue-600 border-blue-200',
        green: 'bg-green-50 text-green-600 border-green-200',
        red: 'bg-red-50 text-red-600 border-red-200',
        yellow: 'bg-yellow-50 text-yellow-600 border-yellow-200',
        gray: 'bg-gray-50 text-gray-600 border-gray-200',
        indigo: 'bg-indigo-50 text-indigo-600 border-indigo-200',
        purple: 'bg-purple-50 text-purple-600 border-purple-200',
        orange: 'bg-orange-50 text-orange-600 border-orange-200',
    };

    return (
        <div className={`rounded-lg border p-5 ${colors[color]}`}>
            <div className="flex items-center justify-between">
                <div>
                    <p className="text-sm font-medium opacity-75">{label}</p>
                    <p className="mt-1 text-2xl font-bold">{value}</p>
                </div>
                <div className="text-3xl opacity-50">{icon}</div>
            </div>
        </div>
    );
};

export default function Dashboard({ stats, recentPayments = [], shops = [] }) {
    return (
        <AuthenticatedLayout
            header={
                <h2 className="text-xl font-semibold leading-tight text-gray-800">
                    Dashboard
                </h2>
            }
        >
            <Head title="Dashboard" />

            <div className="py-6">
                <div className="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8">
                    {/* Summary Cards */}
                    <div className="grid grid-cols-2 gap-4 sm:grid-cols-2 lg:grid-cols-4">
                        <StatCard
                            icon="🏪"
                            label="Total Shops"
                            value={stats.totalShops}
                            color="blue"
                        />
                        <StatCard
                            icon="✅"
                            label="Occupied"
                            value={stats.occupiedShops}
                            color="green"
                        />
                        <StatCard
                            icon="🔑"
                            label="Vacant"
                            value={stats.vacantShops}
                            color="gray"
                        />
                        <StatCard
                            icon="👤"
                            label="Owners"
                            value={stats.totalOwners}
                            color="indigo"
                        />
                        <StatCard
                            icon="📊"
                            label="Monthly Expected"
                            value={formatCurrency(stats.monthlyExpected)}
                            color="blue"
                        />
                        <StatCard
                            icon="💰"
                            label="Collected"
                            value={formatCurrency(stats.monthlyCollected)}
                            color="green"
                        />
                        <StatCard
                            icon="⏳"
                            label="Pending"
                            value={formatCurrency(stats.pendingAmount)}
                            color="yellow"
                        />
                        <StatCard
                            icon="⚠️"
                            label="Overdue Invoices"
                            value={stats.overdueInvoices}
                            color="red"
                        />
                    </div>

                    {/* Security Deposit Dues */}
                    {stats.securityDepositDues > 0 && (
                        <div className="mt-4 rounded-lg border border-orange-200 bg-orange-50 p-4 text-orange-700">
                            <span className="font-semibold">Security Deposit Dues:</span>{' '}
                            {formatCurrency(stats.securityDepositDues)}
                        </div>
                    )}

                    {/* Shops Overview */}
                    <div className="mt-8">
                        <h3 className="mb-4 text-lg font-semibold text-gray-800">Shops Overview</h3>
                        <div className="grid grid-cols-3 gap-3 sm:grid-cols-4 md:grid-cols-6 lg:grid-cols-8">
                            {shops.map((shop) => {
                                const isOccupied = !!shop.active_lease;
                                return (
                                    <Link
                                        key={shop.id}
                                        href={route('shops.show', shop.id)}
                                        className={`rounded-lg border p-3 text-center transition hover:shadow-md ${
                                            isOccupied
                                                ? 'border-green-300 bg-green-50 text-green-800'
                                                : 'border-gray-300 bg-gray-50 text-gray-500'
                                        }`}
                                    >
                                        <div className="text-sm font-bold">{shop.shop_number}</div>
                                        {isOccupied && shop.active_lease?.tenant && (
                                            <div className="mt-1 truncate text-xs">
                                                {shop.active_lease.tenant.name}
                                            </div>
                                        )}
                                        {!isOccupied && (
                                            <div className="mt-1 text-xs italic">Vacant</div>
                                        )}
                                        {shop.owner && (
                                            <div className="mt-1 truncate text-xs opacity-60">
                                                {shop.owner.name}
                                            </div>
                                        )}
                                    </Link>
                                );
                            })}
                        </div>
                    </div>

                    {/* Recent Payments */}
                    <div className="mt-8">
                        <h3 className="mb-4 text-lg font-semibold text-gray-800">Recent Payments</h3>
                        <div className="overflow-hidden rounded-lg bg-white shadow">
                            <div className="overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead className="bg-gray-50">
                                        <tr>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase text-gray-500">Date</th>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase text-gray-500">Tenant</th>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase text-gray-500">Shop</th>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase text-gray-500">Amount</th>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase text-gray-500">Method</th>
                                        </tr>
                                    </thead>
                                    <tbody className="divide-y divide-gray-200">
                                        {recentPayments.length === 0 && (
                                            <tr>
                                                <td colSpan="5" className="px-4 py-6 text-center text-gray-400">
                                                    No recent payments
                                                </td>
                                            </tr>
                                        )}
                                        {recentPayments.map((payment) => (
                                            <tr key={payment.id} className="hover:bg-gray-50">
                                                <td className="whitespace-nowrap px-4 py-3 text-sm text-gray-700">
                                                    {formatDate(payment.payment_date)}
                                                </td>
                                                <td className="whitespace-nowrap px-4 py-3 text-sm text-gray-700">
                                                    {payment.invoice?.lease?.tenant?.name || '-'}
                                                </td>
                                                <td className="whitespace-nowrap px-4 py-3 text-sm text-gray-700">
                                                    {payment.invoice?.lease?.shop?.shop_number || '-'}
                                                </td>
                                                <td className="whitespace-nowrap px-4 py-3 text-sm font-medium text-gray-900">
                                                    {formatCurrency(payment.amount)}
                                                </td>
                                                <td className="whitespace-nowrap px-4 py-3 text-sm text-gray-700 capitalize">
                                                    {payment.payment_method || '-'}
                                                </td>
                                            </tr>
                                        ))}
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AuthenticatedLayout>
    );
}
