import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { Head, Link, useForm, router } from '@inertiajs/react';
import { formatCurrency, formatDate, formatMonth } from '@/utils';

export default function Edit({ reading }) {
    const lease = reading.lease;

    const { data, setData, put, processing, errors } = useForm({
        previous_reading: reading.previous_reading || '',
        current_reading: reading.current_reading || '',
        rate_per_unit: reading.rate_per_unit || '',
    });

    const units = (parseFloat(data.current_reading) || 0) - (parseFloat(data.previous_reading) || 0);
    const amount = units > 0 ? units * (parseFloat(data.rate_per_unit) || 0) : 0;

    const handleSubmit = (e) => {
        e.preventDefault();
        put(route('meter-readings.update', reading.id));
    };

    return (
        <AuthenticatedLayout header={<h2 className="text-xl font-semibold leading-tight text-gray-800">Edit Meter Reading</h2>}>
            <Head title="Edit Meter Reading" />

            <div className="py-12">
                <div className="mx-auto max-w-2xl sm:px-6 lg:px-8">
                    <div className="overflow-hidden bg-white shadow-sm sm:rounded-lg">
                        <div className="p-6 text-gray-900">
                            <div className="flex items-center justify-between mb-6">
                                <h3 className="text-lg font-medium">Edit Meter Reading</h3>
                                <Link href={route('meter-readings.index')} className="inline-flex items-center rounded-md bg-gray-200 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-300">Back</Link>
                            </div>

                            {/* Reading Info */}
                            <div className="rounded-lg border border-gray-200 bg-gray-50 p-4 mb-6">
                                <div className="grid grid-cols-2 gap-4 text-sm">
                                    <div>
                                        <span className="text-gray-500">Shop</span>
                                        <p className="font-medium text-gray-900">Shop {lease?.shop?.shop_number || '—'}</p>
                                    </div>
                                    <div>
                                        <span className="text-gray-500">Tenant</span>
                                        <p className="font-medium text-gray-900">{lease?.tenant?.name || '—'}</p>
                                    </div>
                                    <div>
                                        <span className="text-gray-500">Month</span>
                                        <p className="font-medium text-gray-900">{formatMonth(reading.month)}</p>
                                    </div>
                                </div>
                            </div>

                            <form onSubmit={handleSubmit} className="space-y-6">
                                {/* Previous Reading */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700">Previous Reading <span className="text-red-500">*</span></label>
                                    <input
                                        type="number"
                                        step="0.01"
                                        min="0"
                                        value={data.previous_reading}
                                        onChange={(e) => setData('previous_reading', e.target.value)}
                                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                    />
                                    {errors.previous_reading && <p className="mt-1 text-sm text-red-600">{errors.previous_reading}</p>}
                                </div>

                                {/* Current Reading */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700">Current Reading <span className="text-red-500">*</span></label>
                                    <input
                                        type="number"
                                        step="0.01"
                                        min="0"
                                        value={data.current_reading}
                                        onChange={(e) => setData('current_reading', e.target.value)}
                                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                    />
                                    {errors.current_reading && <p className="mt-1 text-sm text-red-600">{errors.current_reading}</p>}
                                </div>

                                {/* Rate Per Unit */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700">Rate Per Unit (PKR) <span className="text-red-500">*</span></label>
                                    <input
                                        type="number"
                                        step="0.01"
                                        min="0"
                                        value={data.rate_per_unit}
                                        onChange={(e) => setData('rate_per_unit', e.target.value)}
                                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                    />
                                    {errors.rate_per_unit && <p className="mt-1 text-sm text-red-600">{errors.rate_per_unit}</p>}
                                </div>

                                {/* Live Calculation */}
                                <div className="rounded-lg border border-blue-200 bg-blue-50 p-4">
                                    <div className="grid grid-cols-2 gap-4 text-sm">
                                        <div>
                                            <span className="text-gray-500">Units Consumed</span>
                                            <p className="font-semibold text-gray-900">{units > 0 ? units.toFixed(2) : '0.00'}</p>
                                        </div>
                                        <div>
                                            <span className="text-gray-500">Total Amount</span>
                                            <p className="font-bold text-blue-800">{formatCurrency(amount)}</p>
                                        </div>
                                    </div>
                                </div>

                                {/* Submit */}
                                <div className="flex items-center justify-end gap-4">
                                    <Link href={route('meter-readings.index')} className="text-sm text-gray-600 hover:text-gray-900">Cancel</Link>
                                    <button
                                        type="submit"
                                        disabled={processing}
                                        className="inline-flex items-center rounded-md bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700 disabled:opacity-50"
                                    >
                                        {processing ? 'Updating...' : 'Update Reading'}
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </AuthenticatedLayout>
    );
}
