import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import ConfirmDeleteModal from '@/Components/ConfirmDeleteModal';
import FormModal from '@/Components/FormModal';
import { Head, Link, router, useForm, usePage } from '@inertiajs/react';
import { useState, useEffect, useRef } from 'react';

const inputClass = "mt-1 block w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 text-sm";

const emptyOwner = {
    name: '',
    phone: '',
    email: '',
    cnic: '',
    address: '',
    bank_name: '',
    bank_account_no: '',
    notes: '',
};

function OwnerFormFields({ form }) {
    return (
        <>
            {/* Name */}
            <div>
                <label className="block text-sm font-medium text-gray-700">Name <span className="text-red-500">*</span></label>
                <input type="text" value={form.data.name} onChange={e => form.setData('name', e.target.value)} className={inputClass} />
                {form.errors.name && <p className="mt-1 text-sm text-red-600">{form.errors.name}</p>}
            </div>

            {/* Phone */}
            <div>
                <label className="block text-sm font-medium text-gray-700">Phone <span className="text-red-500">*</span></label>
                <input type="text" value={form.data.phone} onChange={e => form.setData('phone', e.target.value)} className={inputClass} />
                {form.errors.phone && <p className="mt-1 text-sm text-red-600">{form.errors.phone}</p>}
            </div>

            {/* Email */}
            <div>
                <label className="block text-sm font-medium text-gray-700">Email</label>
                <input type="email" value={form.data.email} onChange={e => form.setData('email', e.target.value)} className={inputClass} />
                {form.errors.email && <p className="mt-1 text-sm text-red-600">{form.errors.email}</p>}
            </div>

            {/* CNIC */}
            <div>
                <label className="block text-sm font-medium text-gray-700">CNIC</label>
                <input type="text" value={form.data.cnic} onChange={e => form.setData('cnic', e.target.value)} placeholder="00000-0000000-0" className={inputClass} />
                {form.errors.cnic && <p className="mt-1 text-sm text-red-600">{form.errors.cnic}</p>}
            </div>

            {/* Address */}
            <div>
                <label className="block text-sm font-medium text-gray-700">Address</label>
                <textarea rows={2} value={form.data.address} onChange={e => form.setData('address', e.target.value)} className={inputClass} />
                {form.errors.address && <p className="mt-1 text-sm text-red-600">{form.errors.address}</p>}
            </div>

            {/* Bank Name + Account No */}
            <div className="grid grid-cols-2 gap-4">
                <div>
                    <label className="block text-sm font-medium text-gray-700">Bank Name</label>
                    <input type="text" value={form.data.bank_name} onChange={e => form.setData('bank_name', e.target.value)} className={inputClass} />
                    {form.errors.bank_name && <p className="mt-1 text-sm text-red-600">{form.errors.bank_name}</p>}
                </div>
                <div>
                    <label className="block text-sm font-medium text-gray-700">Bank Account No</label>
                    <input type="text" value={form.data.bank_account_no} onChange={e => form.setData('bank_account_no', e.target.value)} className={inputClass} />
                    {form.errors.bank_account_no && <p className="mt-1 text-sm text-red-600">{form.errors.bank_account_no}</p>}
                </div>
            </div>

            {/* Notes */}
            <div>
                <label className="block text-sm font-medium text-gray-700">Notes</label>
                <textarea rows={2} value={form.data.notes} onChange={e => form.setData('notes', e.target.value)} className={inputClass} />
                {form.errors.notes && <p className="mt-1 text-sm text-red-600">{form.errors.notes}</p>}
            </div>
        </>
    );
}

export default function Index({ owners, filters = {} }) {
    const { auth } = usePage().props;
    const [search, setSearch] = useState(filters.search || '');
    const isFirstRender = useRef(true);
    const [deleteId, setDeleteId] = useState(null);
    const [deleting, setDeleting] = useState(false);

    const [showAddModal, setShowAddModal] = useState(false);
    const [editingOwner, setEditingOwner] = useState(null);

    const addForm = useForm({ ...emptyOwner });
    const editForm = useForm({ ...emptyOwner });

    useEffect(() => {
        if (isFirstRender.current) {
            isFirstRender.current = false;
            return;
        }

        const timeout = setTimeout(() => {
            router.get(
                route('owners.index'),
                { search: search || undefined },
                { preserveState: true, replace: true }
            );
        }, 300);

        return () => clearTimeout(timeout);
    }, [search]);

    useEffect(() => {
        if (editingOwner) {
            editForm.setData({
                name: editingOwner.name || '',
                phone: editingOwner.phone || '',
                email: editingOwner.email || '',
                cnic: editingOwner.cnic || '',
                address: editingOwner.address || '',
                bank_name: editingOwner.bank_name || '',
                bank_account_no: editingOwner.bank_account_no || '',
                notes: editingOwner.notes || '',
            });
        }
    }, [editingOwner]);

    const handleDelete = () => {
        setDeleting(true);
        router.delete(route('owners.destroy', deleteId), {
            onFinish: () => { setDeleting(false); setDeleteId(null); },
        });
    };

    const handleAdd = (e) => {
        e.preventDefault();
        addForm.post(route('owners.store'), {
            onSuccess: () => { setShowAddModal(false); addForm.reset(); },
        });
    };

    const handleEdit = (e) => {
        e.preventDefault();
        editForm.put(route('owners.update', editingOwner.id), {
            onSuccess: () => setEditingOwner(null),
        });
    };

    return (
        <AuthenticatedLayout
            header={
                <div className="flex items-center justify-between">
                    <h2 className="text-xl font-semibold leading-tight text-gray-800">
                        Owners
                    </h2>
                    <button
                        type="button"
                        onClick={() => setShowAddModal(true)}
                        className="inline-flex items-center rounded-md bg-blue-600 px-4 py-2 text-sm font-medium text-white shadow-sm hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2"
                    >
                        + Add Owner
                    </button>
                </div>
            }
        >
            <Head title="Owners" />

            <div className="py-6">
                <div className="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8">
                    {/* Search */}
                    <div className="mb-4">
                        <input
                            type="text"
                            placeholder="Search owners by name, phone, or CNIC..."
                            value={search}
                            onChange={(e) => setSearch(e.target.value)}
                            className="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:w-96"
                        />
                    </div>

                    {/* Table */}
                    <div className="overflow-hidden rounded-lg bg-white shadow">
                        <div className="overflow-x-auto">
                            <table className="min-w-full divide-y divide-gray-200">
                                <thead className="bg-gray-50">
                                    <tr>
                                        <th className="px-4 py-3 text-left text-xs font-medium uppercase text-gray-500">Name</th>
                                        <th className="px-4 py-3 text-left text-xs font-medium uppercase text-gray-500">Phone</th>
                                        <th className="px-4 py-3 text-left text-xs font-medium uppercase text-gray-500">CNIC</th>
                                        <th className="px-4 py-3 text-left text-xs font-medium uppercase text-gray-500">Shops</th>
                                        <th className="px-4 py-3 text-right text-xs font-medium uppercase text-gray-500">Actions</th>
                                    </tr>
                                </thead>
                                <tbody className="divide-y divide-gray-200">
                                    {owners.data.length === 0 && (
                                        <tr>
                                            <td colSpan="5" className="px-4 py-6 text-center text-gray-400">
                                                No owners found.
                                            </td>
                                        </tr>
                                    )}
                                    {owners.data.map((owner) => (
                                        <tr key={owner.id} className="hover:bg-gray-50">
                                            <td className="whitespace-nowrap px-4 py-3 text-sm font-medium text-gray-900">
                                                {owner.name}
                                            </td>
                                            <td className="whitespace-nowrap px-4 py-3 text-sm text-gray-700">
                                                {owner.phone || '-'}
                                            </td>
                                            <td className="whitespace-nowrap px-4 py-3 text-sm text-gray-700">
                                                {owner.cnic || '-'}
                                            </td>
                                            <td className="whitespace-nowrap px-4 py-3 text-sm text-gray-700">
                                                {owner.shops_count ?? owner.shops?.length ?? 0}
                                            </td>
                                            <td className="whitespace-nowrap px-4 py-3 text-right text-sm">
                                                <Link
                                                    href={route('owners.show', owner.id)}
                                                    className="mr-2 text-blue-600 hover:text-blue-800"
                                                >
                                                    View
                                                </Link>
                                                {auth.user.is_admin && (
                                                <button
                                                    type="button"
                                                    onClick={() => setEditingOwner(owner)}
                                                    className="mr-2 text-indigo-600 hover:text-indigo-800"
                                                >
                                                    Edit
                                                </button>
                                                )}
                                                {auth.user.is_admin && (
                                                <button
                                                    onClick={() => setDeleteId(owner.id)}
                                                    className="text-red-600 hover:text-red-800"
                                                >
                                                    Delete
                                                </button>
                                                )}
                                            </td>
                                        </tr>
                                    ))}
                                </tbody>
                            </table>
                        </div>
                    </div>

                    {/* Pagination */}
                    {owners.links && (
                        <div className="mt-4 flex flex-wrap items-center justify-center gap-1">
                            {owners.links.map((link, i) => (
                                <Link
                                    key={i}
                                    href={link.url || '#'}
                                    className={`rounded px-3 py-1 text-sm ${
                                        link.active
                                            ? 'bg-blue-600 text-white'
                                            : link.url
                                            ? 'bg-gray-200 text-gray-700 hover:bg-gray-300'
                                            : 'cursor-not-allowed bg-gray-100 text-gray-400'
                                    }`}
                                    dangerouslySetInnerHTML={{ __html: link.label }}
                                    preserveState
                                />
                            ))}
                        </div>
                    )}
                </div>
            </div>

            {/* Delete Modal */}
            <ConfirmDeleteModal
                show={deleteId !== null}
                onClose={() => setDeleteId(null)}
                onConfirm={handleDelete}
                title="Delete Owner"
                message="Are you sure you want to delete this owner? All associated data will be removed."
                processing={deleting}
            />

            {/* Add Modal */}
            <FormModal show={showAddModal} onClose={() => setShowAddModal(false)} title="Add New Owner">
                <form onSubmit={handleAdd} className="space-y-4">
                    <OwnerFormFields form={addForm} />
                    <div className="flex justify-end gap-3 pt-4 border-t">
                        <button type="button" onClick={() => setShowAddModal(false)} className="inline-flex items-center rounded-md bg-gray-100 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-200 focus:outline-none focus:ring-2 focus:ring-gray-300">
                            Cancel
                        </button>
                        <button type="submit" disabled={addForm.processing} className="inline-flex items-center rounded-md bg-blue-600 px-4 py-2 text-sm font-medium text-white shadow-sm hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 disabled:opacity-50">
                            Save Owner
                        </button>
                    </div>
                </form>
            </FormModal>

            {/* Edit Modal */}
            <FormModal show={editingOwner !== null} onClose={() => setEditingOwner(null)} title="Edit Owner">
                <form onSubmit={handleEdit} className="space-y-4">
                    <OwnerFormFields form={editForm} />
                    <div className="flex justify-end gap-3 pt-4 border-t">
                        <button type="button" onClick={() => setEditingOwner(null)} className="inline-flex items-center rounded-md bg-gray-100 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-200 focus:outline-none focus:ring-2 focus:ring-gray-300">
                            Cancel
                        </button>
                        <button type="submit" disabled={editForm.processing} className="inline-flex items-center rounded-md bg-blue-600 px-4 py-2 text-sm font-medium text-white shadow-sm hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 disabled:opacity-50">
                            Update Owner
                        </button>
                    </div>
                </form>
            </FormModal>
        </AuthenticatedLayout>
    );
}
