import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import ConfirmDeleteModal from '@/Components/ConfirmDeleteModal';
import FormModal from '@/Components/FormModal';
import { Head, Link, useForm, router, usePage } from '@inertiajs/react';
import { formatCurrency, formatDate } from '@/utils';
import { useState, useEffect } from 'react';

export default function Index({ shops, owners, filters }) {
    const { auth } = usePage().props;
    const [deleteId, setDeleteId] = useState(null);
    const [deleting, setDeleting] = useState(false);
    const [showAddModal, setShowAddModal] = useState(false);
    const [editingShop, setEditingShop] = useState(null);

    const addForm = useForm({ owner_id: '', shop_number: '', description: '', location: '', area_sqft: '', default_rent: '' });
    const editForm = useForm({ owner_id: '', shop_number: '', description: '', location: '', area_sqft: '', default_rent: '' });

    useEffect(() => {
        if (editingShop) {
            editForm.setData({
                owner_id: editingShop.owner_id || '',
                shop_number: editingShop.shop_number || '',
                description: editingShop.description || '',
                location: editingShop.location || '',
                area_sqft: editingShop.area_sqft || '',
                default_rent: editingShop.default_rent || '',
            });
        }
    }, [editingShop]);

    const applyFilters = (newFilters) => {
        router.get(route('shops.index'), { ...filters, ...newFilters }, { preserveState: true, replace: true });
    };

    const handleDelete = () => {
        setDeleting(true);
        router.delete(route('shops.destroy', deleteId), {
            onFinish: () => { setDeleting(false); setDeleteId(null); },
        });
    };

    const handleAddSubmit = (e) => {
        e.preventDefault();
        addForm.post(route('shops.store'), {
            onSuccess: () => { setShowAddModal(false); addForm.reset(); },
        });
    };

    const handleEditSubmit = (e) => {
        e.preventDefault();
        editForm.put(route('shops.update', editingShop.id), {
            onSuccess: () => setEditingShop(null),
        });
    };

    const renderFormFields = (form) => (
        <>
            <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Owner <span className="text-red-500">*</span></label>
                <select
                    value={form.data.owner_id}
                    onChange={(e) => form.setData('owner_id', e.target.value)}
                    className="w-full rounded-lg border-gray-300 text-sm shadow-sm focus:border-blue-500 focus:ring-blue-500"
                >
                    <option value="">Select Owner</option>
                    {owners.map((owner) => (
                        <option key={owner.id} value={owner.id}>{owner.name}</option>
                    ))}
                </select>
                {form.errors.owner_id && <p className="mt-1 text-xs text-red-600">{form.errors.owner_id}</p>}
            </div>
            <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Shop Number <span className="text-red-500">*</span></label>
                <input
                    type="text"
                    value={form.data.shop_number}
                    onChange={(e) => form.setData('shop_number', e.target.value)}
                    className="w-full rounded-lg border-gray-300 text-sm shadow-sm focus:border-blue-500 focus:ring-blue-500"
                />
                {form.errors.shop_number && <p className="mt-1 text-xs text-red-600">{form.errors.shop_number}</p>}
            </div>
            <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Description</label>
                <textarea
                    rows={2}
                    value={form.data.description}
                    onChange={(e) => form.setData('description', e.target.value)}
                    className="w-full rounded-lg border-gray-300 text-sm shadow-sm focus:border-blue-500 focus:ring-blue-500"
                />
                {form.errors.description && <p className="mt-1 text-xs text-red-600">{form.errors.description}</p>}
            </div>
            <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Location</label>
                <input
                    type="text"
                    value={form.data.location}
                    onChange={(e) => form.setData('location', e.target.value)}
                    className="w-full rounded-lg border-gray-300 text-sm shadow-sm focus:border-blue-500 focus:ring-blue-500"
                />
                {form.errors.location && <p className="mt-1 text-xs text-red-600">{form.errors.location}</p>}
            </div>
            <div className="grid grid-cols-2 gap-4">
                <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Area (sqft)</label>
                    <input
                        type="number"
                        value={form.data.area_sqft}
                        onChange={(e) => form.setData('area_sqft', e.target.value)}
                        className="w-full rounded-lg border-gray-300 text-sm shadow-sm focus:border-blue-500 focus:ring-blue-500"
                    />
                    {form.errors.area_sqft && <p className="mt-1 text-xs text-red-600">{form.errors.area_sqft}</p>}
                </div>
                <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Default Rent</label>
                    <input
                        type="number"
                        value={form.data.default_rent}
                        onChange={(e) => form.setData('default_rent', e.target.value)}
                        className="w-full rounded-lg border-gray-300 text-sm shadow-sm focus:border-blue-500 focus:ring-blue-500"
                    />
                    {form.errors.default_rent && <p className="mt-1 text-xs text-red-600">{form.errors.default_rent}</p>}
                </div>
            </div>
        </>
    );

    return (
        <AuthenticatedLayout header={<h2 className="text-xl font-semibold leading-tight text-gray-800">Shops</h2>}>
            <Head title="Shops" />

            <div className="py-12">
                <div className="mx-auto max-w-7xl sm:px-6 lg:px-8">
                    <div className="overflow-hidden bg-white shadow-sm sm:rounded-lg">
                        <div className="p-6 text-gray-900">
                            {/* Header */}
                            <div className="flex items-center justify-between mb-6">
                                <h3 className="text-lg font-medium">All Shops</h3>
                                <button
                                    onClick={() => setShowAddModal(true)}
                                    className="inline-flex items-center rounded-md bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
                                >
                                    Add Shop
                                </button>
                            </div>

                            {/* Filters */}
                            <div className="mb-6 grid grid-cols-1 gap-4 sm:grid-cols-3">
                                <input
                                    type="text"
                                    placeholder="Search shops..."
                                    defaultValue={filters.search}
                                    onChange={(e) => applyFilters({ search: e.target.value })}
                                    className="rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                />
                                <select
                                    defaultValue={filters.owner_id || ''}
                                    onChange={(e) => applyFilters({ owner_id: e.target.value })}
                                    className="rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                >
                                    <option value="">All Owners</option>
                                    {owners.map((owner) => (
                                        <option key={owner.id} value={owner.id}>{owner.name}</option>
                                    ))}
                                </select>
                                <select
                                    defaultValue={filters.status || ''}
                                    onChange={(e) => applyFilters({ status: e.target.value })}
                                    className="rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                >
                                    <option value="">All Status</option>
                                    <option value="vacant">Vacant</option>
                                    <option value="occupied">Occupied</option>
                                </select>
                            </div>

                            {/* Table */}
                            <div className="overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead className="bg-gray-50">
                                        <tr>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Shop #</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Owner</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Location</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Default Rent</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Status</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Current Tenant</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody className="divide-y divide-gray-200 bg-white">
                                        {shops.data.map((shop) => (
                                            <tr key={shop.id} className="hover:bg-gray-50">
                                                <td className="whitespace-nowrap px-6 py-4 text-sm font-medium text-gray-900">{shop.shop_number}</td>
                                                <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-500">{shop.owner?.name}</td>
                                                <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-500">{shop.location}</td>
                                                <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-500">{formatCurrency(shop.default_rent)}</td>
                                                <td className="whitespace-nowrap px-6 py-4 text-sm">
                                                    <span className={`inline-flex rounded-full px-2 text-xs font-semibold leading-5 ${
                                                        shop.active_lease
                                                            ? 'bg-green-100 text-green-800'
                                                            : 'bg-gray-100 text-gray-800'
                                                    }`}>
                                                        {shop.active_lease ? 'Occupied' : 'Vacant'}
                                                    </span>
                                                </td>
                                                <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-500">
                                                    {shop.active_lease?.tenant?.name || '—'}
                                                </td>
                                                <td className="whitespace-nowrap px-6 py-4 text-sm space-x-2">
                                                    <Link href={route('shops.show', shop.id)} className="text-blue-600 hover:text-blue-900">View</Link>
                                                    {auth.user.is_admin && (
                                                    <button onClick={() => setEditingShop(shop)} className="text-indigo-600 hover:text-indigo-900">Edit</button>
                                                    )}
                                                    {auth.user.is_admin && (
                                                    <button onClick={() => setDeleteId(shop.id)} className="text-red-600 hover:text-red-900">Delete</button>
                                                    )}
                                                </td>
                                            </tr>
                                        ))}
                                        {shops.data.length === 0 && (
                                            <tr>
                                                <td colSpan="7" className="px-6 py-4 text-center text-sm text-gray-500">No shops found.</td>
                                            </tr>
                                        )}
                                    </tbody>
                                </table>
                            </div>

                            {/* Pagination */}
                            {shops.links && shops.links.length > 3 && (
                                <div className="flex flex-wrap items-center gap-1 mt-4">
                                    {shops.links.map((link, i) => (
                                        link.url ? (
                                            <Link key={i} href={link.url} className={`px-3 py-1 text-sm rounded ${link.active ? 'bg-blue-600 text-white' : 'bg-white border hover:bg-gray-50'}`}
                                                dangerouslySetInnerHTML={{ __html: link.label }} preserveState />
                                        ) : (
                                            <span key={i} className="px-3 py-1 text-sm text-gray-400" dangerouslySetInnerHTML={{ __html: link.label }} />
                                        )
                                    ))}
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>

            {/* Add Shop Modal */}
            <FormModal show={showAddModal} onClose={() => setShowAddModal(false)} title="Add Shop">
                <form onSubmit={handleAddSubmit} className="space-y-4">
                    {renderFormFields(addForm)}
                    <div className="flex justify-end gap-3 border-t border-gray-100 pt-4">
                        <button type="button" onClick={() => setShowAddModal(false)} className="rounded-lg border border-gray-300 bg-white px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50">Cancel</button>
                        <button type="submit" disabled={addForm.processing} className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700 disabled:opacity-50">Save</button>
                    </div>
                </form>
            </FormModal>

            {/* Edit Shop Modal */}
            <FormModal show={editingShop !== null} onClose={() => setEditingShop(null)} title="Edit Shop">
                <form onSubmit={handleEditSubmit} className="space-y-4">
                    {renderFormFields(editForm)}
                    <div className="flex justify-end gap-3 border-t border-gray-100 pt-4">
                        <button type="button" onClick={() => setEditingShop(null)} className="rounded-lg border border-gray-300 bg-white px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50">Cancel</button>
                        <button type="submit" disabled={editForm.processing} className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700 disabled:opacity-50">Update</button>
                    </div>
                </form>
            </FormModal>

            <ConfirmDeleteModal
                show={deleteId !== null}
                onClose={() => setDeleteId(null)}
                onConfirm={handleDelete}
                title="Delete Shop"
                message="Are you sure you want to delete this shop? All related leases and invoices will be affected."
                processing={deleting}
            />
        </AuthenticatedLayout>
    );
}
