import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { Head, Link } from '@inertiajs/react';
import { formatCurrency, formatDate } from '@/utils';

export default function Show({ shop }) {
    return (
        <AuthenticatedLayout header={<h2 className="text-xl font-semibold leading-tight text-gray-800">Shop Details</h2>}>
            <Head title={`Shop ${shop.shop_number}`} />

            <div className="py-12">
                <div className="mx-auto max-w-7xl sm:px-6 lg:px-8 space-y-6">
                    {/* Shop Details Card */}
                    <div className="overflow-hidden bg-white shadow-sm sm:rounded-lg">
                        <div className="p-6 text-gray-900">
                            <div className="flex items-center justify-between mb-6">
                                <h3 className="text-lg font-medium">Shop #{shop.shop_number}</h3>
                                <div className="space-x-2">
                                    <Link href={route('shops.edit', shop.id)} className="inline-flex items-center rounded-md bg-indigo-600 px-4 py-2 text-sm font-medium text-white hover:bg-indigo-700">Edit</Link>
                                    <Link href={route('shops.index')} className="inline-flex items-center rounded-md bg-gray-200 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-300">Back</Link>
                                </div>
                            </div>

                            <dl className="grid grid-cols-1 gap-x-4 gap-y-6 sm:grid-cols-2 lg:grid-cols-3">
                                <div>
                                    <dt className="text-sm font-medium text-gray-500">Shop Number</dt>
                                    <dd className="mt-1 text-sm text-gray-900">{shop.shop_number}</dd>
                                </div>
                                <div>
                                    <dt className="text-sm font-medium text-gray-500">Owner</dt>
                                    <dd className="mt-1 text-sm text-gray-900">{shop.owner?.name}</dd>
                                </div>
                                <div>
                                    <dt className="text-sm font-medium text-gray-500">Location</dt>
                                    <dd className="mt-1 text-sm text-gray-900">{shop.location || '—'}</dd>
                                </div>
                                <div>
                                    <dt className="text-sm font-medium text-gray-500">Area (sq ft)</dt>
                                    <dd className="mt-1 text-sm text-gray-900">{shop.area_sqft || '—'}</dd>
                                </div>
                                <div>
                                    <dt className="text-sm font-medium text-gray-500">Default Rent</dt>
                                    <dd className="mt-1 text-sm text-gray-900">{formatCurrency(shop.default_rent)}</dd>
                                </div>
                                <div>
                                    <dt className="text-sm font-medium text-gray-500">Status</dt>
                                    <dd className="mt-1">
                                        <span className={`inline-flex rounded-full px-2 text-xs font-semibold leading-5 ${
                                            shop.active_lease
                                                ? 'bg-green-100 text-green-800'
                                                : 'bg-gray-100 text-gray-800'
                                        }`}>
                                            {shop.active_lease ? 'Occupied' : 'Vacant'}
                                        </span>
                                    </dd>
                                </div>
                            </dl>

                            {shop.description && (
                                <div className="mt-6">
                                    <dt className="text-sm font-medium text-gray-500">Description</dt>
                                    <dd className="mt-1 text-sm text-gray-900">{shop.description}</dd>
                                </div>
                            )}
                        </div>
                    </div>

                    {/* Current Tenant */}
                    {shop.active_lease && (
                        <div className="overflow-hidden bg-white shadow-sm sm:rounded-lg">
                            <div className="p-6 text-gray-900">
                                <h3 className="text-lg font-medium mb-4">Current Tenant</h3>
                                <dl className="grid grid-cols-1 gap-x-4 gap-y-4 sm:grid-cols-3">
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Tenant Name</dt>
                                        <dd className="mt-1 text-sm text-gray-900">
                                            <Link href={route('tenants.show', shop.active_lease.tenant?.id)} className="text-blue-600 hover:text-blue-900">
                                                {shop.active_lease.tenant?.name}
                                            </Link>
                                        </dd>
                                    </div>
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Lease Start</dt>
                                        <dd className="mt-1 text-sm text-gray-900">{formatDate(shop.active_lease.start_date)}</dd>
                                    </div>
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Monthly Rent</dt>
                                        <dd className="mt-1 text-sm text-gray-900">{formatCurrency(shop.active_lease.rent)}</dd>
                                    </div>
                                </dl>
                            </div>
                        </div>
                    )}

                    {/* Lease History */}
                    <div className="overflow-hidden bg-white shadow-sm sm:rounded-lg">
                        <div className="p-6 text-gray-900">
                            <h3 className="text-lg font-medium mb-4">Lease History</h3>
                            <div className="overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead className="bg-gray-50">
                                        <tr>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Tenant</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Start Date</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">End Date</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Rent</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Status</th>
                                        </tr>
                                    </thead>
                                    <tbody className="divide-y divide-gray-200 bg-white">
                                        {shop.leases?.map((lease) => (
                                            <tr key={lease.id} className="hover:bg-gray-50">
                                                <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-900">{lease.tenant?.name}</td>
                                                <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-500">{formatDate(lease.start_date)}</td>
                                                <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-500">{lease.end_date ? formatDate(lease.end_date) : '—'}</td>
                                                <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-500">{formatCurrency(lease.rent)}</td>
                                                <td className="whitespace-nowrap px-6 py-4 text-sm">
                                                    <span className={`inline-flex rounded-full px-2 text-xs font-semibold leading-5 ${
                                                        lease.status === 'active'
                                                            ? 'bg-green-100 text-green-800'
                                                            : 'bg-gray-100 text-gray-800'
                                                    }`}>
                                                        {lease.status}
                                                    </span>
                                                </td>
                                            </tr>
                                        ))}
                                        {(!shop.leases || shop.leases.length === 0) && (
                                            <tr>
                                                <td colSpan="5" className="px-6 py-4 text-center text-sm text-gray-500">No lease history.</td>
                                            </tr>
                                        )}
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AuthenticatedLayout>
    );
}
